function [] = analyze_gyro_data_v8()
% Main Data processing code for gyro data collected with ZI MFLI Lock-in
% Amplifier.  

% Last update August 2020

% Copyright - A. Taranta 2024

close all
clear all

%% - REST OF CODE


h_ind = 1;
force_process_data = 0; %if false, the "vars.mat" file will be loaded (if available) instead of raw data           
use_thermal_data = 0;   %Flag determining whether or not to use thermal data
show_all_Ts = 0;        %Flag determing wether or not to show all the temperatures
fit_T_dot = 0;          %Flag determining whether or not to do T-dot fitting
make_COMSOL_T_file = 0; %Flag for whether or not to create a csv file with the temperature for model simulation use
sv_pngs = 1;            %Flag for whether or not to save png pictures of the plotted data
no_light_flag = true;   %flag true if analyzing no light data, and X and R raw data will have fake signal added 
fake_f1X = 12.5693e-6;     %Fake value of X for 1st harmonic demod
fake_f2R = 67.0401e-3;     %Fake value of R for 2nd harmonic demod

is_power_sweep = 0;     %Not a power sweep by default
show_int_vs_T = 0;      %Flag to show intensity vs. temperature among t-dot plots
use_true_SF = 1; %boolean using calculated vs measure SF
%mod_depth = 3*pi/4; %modulation depth in Radians

signal_processing = true; %dont do advanced signal processing if false (Allan Dev takes forever)

Allan_sec = true;


lat = 50.937615; %B53 Latitude
true_rate = 15.*sin(deg2rad(lat)); %True Rotation Rate at B53 latitude for IA VERTICAL

horz_figs = [0.9 0.5];
vert_figs = [0.5 0.9];

%% - SETUP FOR WHICH DATA IS SHOWN

main_dir = 'D:\Work\FOG Development\Test Data\MCIFOG 1 (150m)\Gyro Data\';
% main_dir = 'C:\Users\Austin\Documents\Work\FOG Development\Test Data\MCIFOG 1\';
% main_dir = '\\ORC.FILES.SOTON.AC.UK\ORCResearch\Q-preforms\FOG Development\Test Data\PM Gyro Build\_Data\On Build Plate\';


% SF_true = 0.951264232; %Heading swing measured scale factor for PM Reference Gyro
%SF_true = 1.55120675239269; %MCIFOG 1 Heading swing measured scale factor at 3pi/4: "20190530 Heading Swing and Overnight Bias (10Hz) at 0.75pi mod depth"
 SF_true = 1.52913490761389; %MCIFOG 1 Heading swing measured scale factor at pi/2: "20190530 Heading Swing"
   
load_FOG_file_settings()

%% - Full Code

data_dir = [main_dir data_dir];

scl = besselj(1,mod_depth)./besselj(2,mod_depth);
applied_SF = 3600*180/pi/SF_true/scl; %scale factor to get rate in deg/hr by multiplying demod word

cd(data_dir);

T_data = [];
if use_thermal_data
    T_data = process_T_log_NEW(data_dir);
end

[filelist, n_files] = get_data_files('stream',data_dir);

rate_demod = 1; %index of rate (1st harmonic) demodulator
int_demod = 4; %index of intensity (2nd harmonic) demodulator

if force_process_data  || exist('vars.mat') ~= 2
    [data,cfg,text_sum] = preprocess_gyro_data_v2(filelist,rate_demod,int_demod);
    save('vars.mat','T_data','data','cfg','text_sum')
else 
    load('vars.mat')
end

tit_str = cell(3,1);
tit_str{1,1} = config_str;
tit_str(2:end,1) = text_sum(1:2);

sr = cfg.sample_rate;
dt = 1./sr; %sample rate in sec
t = data.t;
t_sec= t;

t_tot = max(t_sec)-min(t_sec);

if t_tot > 2*60*60*24 
        xlab_str = 'Elapsed Time (Days)';
        time_units = 'd';
        time_scl = 1/(60*60*24);
elseif t_tot > 2*60*60
        xlab_str = 'Elapsed Time (Hours)';
        time_units = 'h';
        time_scl = 1/(60*60);
elseif t_tot > 2*60
        xlab_str = 'Elapsed Time (Min.)';
        time_units = 'm';
        time_scl = 1/60;
else 
        xlab_str = 'Elapsed Time (Sec.)';
        time_units = 's';
        time_scl = 1;
end

t_disp = time_scl.*t_sec;
t_disp_all = t_disp;



%m_time = datetime(LV_time,'ConvertFrom','epochTime','Epoch','1904-01-01','TimeZone','UTC');

if kill_data
    keepers1 = find(t_disp>t_min & t_disp < kill_lo);
    keepers2 = find(t_disp>kill_hi & t_disp < t_max);
    keepers = [keepers1; keepers2];
else
    keepers = find(t_disp>t_min & t_disp < t_max);
end

t_sec = t_sec(keepers);


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Create new folder for housing the figures that correspond to this time
% selection for the present run
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if t_min > 0
    tmin_str = [sprintf('%g',t_min) time_units];
else
    tmin_str = 'Start';
end

if t_max < t_disp_all(end)
    tmax_str = [sprintf('%g',t_max) time_units];
else
    tmax_str = 'End';
end

if kill_data
    kill_str = [sprintf(' - %g to %g',kill_lo,kill_hi) time_units ' Removed'];
else
    kill_str = [];
end

new_fig_folder = [tmin_str ' to ' tmax_str kill_str];

if exist(new_fig_folder) ~= 7
    mkdir(new_fig_folder)
end

cd(new_fig_folder)


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% REVISITED SO THAT WE CAN CHANGE SCALE ON SMALL RUNS
t_tot = max(t_sec)-min(t_sec);

if t_tot > 2*60*60*24 
        xlab_str = 'Elapsed Time (Days)';
        time_units = 'd';
        time_scl = 1/(60*60*24);
elseif t_tot > 2*60*60
        xlab_str = 'Elapsed Time (Hours)';
        time_units = 'h';
        time_scl = 1/(60*60);
elseif t_tot > 2*60
        xlab_str = 'Elapsed Time (Min.)';
        time_units = 'm';
        time_scl = 1/60;
else 
        xlab_str = 'Elapsed Time (Sec.)';
        time_units = 's';
        time_scl = 1;
end

t_disp = time_scl.*t_sec;
t_disp_all = t_disp;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if use_thermal_data
    t_T = seconds(T_data.time - T_data.time(1));
    t_lag = 0;
    t_T = t_T - t_lag;
   
    heater_output       = T_data.Heater; %interp1(t_T, T_data.Heater     ,t_sec,'linear');
    coil_T_interp       = interp1(t_T, T_data.T_coil     ,t_sec,'linear');
    igloo_T_interp      = interp1(t_T, T_data.T_igloo    ,t_sec,'linear');
    optics_T_interp     = interp1(t_T, T_data.T_optics   ,t_sec,'linear');
    lab_T_interp        = interp1(t_T, T_data.T_lab      ,t_sec,'linear');
    plate_T_interp      = interp1(t_T, T_data.T_plate    ,t_sec,'linear');
    air_T_interp        = interp1(t_T, T_data.T_air      ,t_sec,'linear');
    t_T = t_T - t_T(1);
    t_T_disp = time_scl.*t_T; 
    
    
    % Temperature used for fitting and main plots
    main_T = plate_T_interp;
end

t_sec = t_sec-t_sec(1);
t_disp = time_scl.*t_sec;


% coil_T_interp;
% igloo_T_interp;
% optics_T_interp;
% lab_T_interp;
% plate_T_interp;
% air_T_interp;

 

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

N = length(t_sec);

rate_quaderr = data.yf1(keepers);   %Raw Rate data - Y from 1st harmonic demod
rate_demod_freq = data.f1(keepers); %1st harmonic Demodulation Frequency
int_demod_freq = data.f2(keepers);  %2nd harmonic Demodulation Frequency
int_x = data.xf2(keepers);          %Intensity - X from 2nd harmonic
int_y = data.yf2(keepers);          %Intensity - Y from 2nd harmonic

if no_light_flag 
    raw_rate = data.xf1(keepers)+fake_f1X;       
    int = sqrt((int_x.^2)+(int_y.^2)+(fake_f2R.^2));  %Intensity - R
else
    raw_rate = data.xf1(keepers);       
    int = sqrt((int_x.^2)+(int_y.^2));  %Intensity - R
end
    
int_log = 10*log10(int);            %Intensity - R expressed in log space
int_diff =  diff(int_log);          %Pointwise derivative of intensity

sample_sep = 300;                   %Peaks must be separated by 300 seconds
sep = round(sr*sample_sep,0);       %minimum interval ()in units of samples) between power splits

%hold on; plot(locs,ones(length(locs)),'ro')

% use_thermal_data = 0;   %Flag determining whether or not to use thermal data
% show_all_Ts = 0;        %Flag determing wether or not to show all the temperatures

clear data

figure
yyaxis left
plot(t_disp,raw_rate.*1e6);
ylabel('Raw Rate Demod - 1st Harmonic X (V)','fontweight','bold')

yyaxis right
plot(t_disp,int.*1e3);
ylabel('Raw Intensity Demod - 2nd Harmonic R (mV)','fontweight','bold')


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Fractional Rate and filtered
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

R = raw_rate./int;
raw_quaderr = rate_quaderr./int;

ds_rate = 1/36; %downsample to 36 sec
win_sz = round(sr./ds_rate); %number of samples to decimate
win_half = floor(win_sz/2); %half size
mv_avg_num = (1/win_sz)*ones(1,win_sz); %numerator for moving avg filter

R_ds = filter(mv_avg_num,1,R);  %averaged data
R_ds = R_ds(win_sz:end);        %re-centered to remove rise

t_ds = t_disp(win_half:end-win_half);
t_ds = t_ds(1:length(R_ds));

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% True Rate Calculation
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if use_true_SF
    rate = R.*applied_SF;
    quaderr = raw_quaderr.*applied_SF;
    rate_no_norm = raw_rate./mean(raw_rate).*mean(rate);
else
    estimated_scale_factor = true_rate./mean(R);    
    rate = R.*estimated_scale_factor;
    quaderr = raw_quaderr.*estimated_scale_factor;
    rate_no_norm = raw_rate./mean(raw_rate).*mean(rate);
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Fractional Rate and filtered
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

rate_filt = filter(mv_avg_num,1,rate); %averaged data
rate_filt = rate_filt(win_sz:end); %re-centered to remove rise

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Filtered Quadrqature Error and Intensity
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
quad_filt = filter(mv_avg_num,1,quaderr); %averaged data
quad_filt = quad_filt(win_sz:end); %re-centered to remove rise

int_filt = filter(mv_avg_num,1,int); %averaged data
int_filt = int_filt(win_sz:end); %re-centered to remove rise

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Fractional Rate and filtered
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if use_thermal_data
    
    T_filt = filter(mv_avg_num,1,main_T); %averaged data
    T_filt = T_filt(win_sz:end); %re-centered to remove rise
    
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% T-Dot Fitting
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if use_thermal_data & fit_T_dot
    
    T0_lsq = ones(length(T_filt)-1,1); %ones
    T_lsq = T_filt(1:end-1); %T in C
    T_dot_lsq = diff(T_filt)./(dt/60); %T-dot in C/min.
    A = [T0_lsq T_lsq T_dot_lsq];
    T_coeffs = (A'*A)\(A'*rate_filt(1:end-1));
    
    T_fit_rate = A*T_coeffs;
    T_fit_rate = [T_fit_rate; T_fit_rate(end)];
    T_dot_lsq = [T_dot_lsq; T_dot_lsq(end)];
    T_fit_res = rate_filt - T_fit_rate;
    
    T0_lsq = ones(length(T_filt)-2,1); %ones
    T_lsq = T_filt(1:end-2); %T in C
    T_dot_lsq = diff(T_filt)./(dt/60); %T-dot in C/min.
    T_dd_lsq = diff(T_dot_lsq)./(dt/60); %T-dot in C/min.
    T_dot_lsq = T_dot_lsq(1:end-1);
    A = [T0_lsq T_lsq T_dot_lsq T_dd_lsq];
    Tdd_coeffs = (A'*A)\(A'*rate_filt(1:end-2));
    
    Tdd_fit_rate = A*Tdd_coeffs;
    
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Allan Deviation
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if signal_processing
%[tau,adev_rate] = gyro_allan_dev(rate,dt./3600,500);
[tau,adev_normrate] = gyro_allan_dev(rate,dt./3600,500);
[tau,adev_rate] = gyro_allan_dev(rate_no_norm,dt./3600,500);

if fit_T_dot
    [tau_filt,adev_rate_filt] = gyro_allan_dev(rate_filt,dt./3600,500);
    [tau_filt,adev_fit_res] = gyro_allan_dev(T_fit_res,dt./3600,500);
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Temperature Allan Dev
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%\
if use_thermal_data
    
    [~    ,adev_coil]   = gyro_allan_dev(coil_T_interp,dt./3600,500);
    [~    ,adev_igloo]  = gyro_allan_dev(igloo_T_interp,dt./3600,500);
    [~    ,adev_optics] = gyro_allan_dev(optics_T_interp,dt./3600,500);
    [~    ,adev_air]    = gyro_allan_dev(lab_T_interp,dt./3600,500);
    [~    ,adev_plate]  = gyro_allan_dev(plate_T_interp,dt./3600,500);
    [tau_T,adev_enc]    = gyro_allan_dev(air_T_interp,dt./3600,500);
    
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Rate & Temperature PSDs
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%\

[freq,psd_rate] = power_spec_fft(rate,sr);
[~,psd_1stharm] = power_spec_fft(raw_rate,sr);
[~,psd_2ndharm] = power_spec_fft(int,sr);

if use_thermal_data
    [~,psd_coil]        = power_spec_fft(coil_T_interp,sr);
    [~,psd_igloo]       = power_spec_fft(igloo_T_interp,sr);
    [~,psd_optics]      = power_spec_fft(optics_T_interp,sr);
    [~,psd_air]         = power_spec_fft(lab_T_interp,sr);
    [~,psd_plate]       = power_spec_fft(plate_T_interp,sr);
    [freq_T,psd_enc]      = power_spec_fft(air_T_interp,sr);
end

end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% POWER SPLIT PROCESSING
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if is_power_sweep
    
    [~,power_splits] = findpeaks(abs(int_diff),'MinPeakProminence',int_diff_thresh,'MinPeakDistance',sep);  % DATAPOINTS AT WHICH POWER CHANGED DRASTICALLY
    n_splits = length(power_splits);
    cols = jet(num_pwrs);
    start_pts = [1;power_splits];

    ind_sep = round(power_track_margin*sr,0); %number of samples to remove from front and back of each power segment
    

    t_disp_seg      = cell(1,1); %segment time
    int_seg         = cell(1,1); %segment 2nd harmonic intensity
    raw_rate_seg    = cell(1,1); %segment 1st harmonic X Trace
    rate_seg        = cell(1,1); %segment normalized rate
    filt_rate_seg   = cell(1,1); %segment filtered (36 sec moving avg) rate
    tau_seg         = cell(1,1); %segment cluster times
    adev_seg        = cell(1,1); %segment rate allan deviation
    
    
    [tau,adev_rate] = gyro_allan_dev(rate_no_norm,dt./3600,500);
    
    for i = 1:n_splits
tic
        k_s = start_pts(i)+ind_sep;
        k_e = start_pts(i+1)-ind_sep;
        
        t_disp_seg{i}       = t_disp(k_s:k_e);
        int_seg{i}          = int(k_s:k_e); 
        raw_rate_seg{i}     = raw_rate(k_s:k_e);
        rate_seg{i}         = rate(k_s:k_e);
        filt_rate_seg{i}    = rate_filt(k_s:k_e);
      
        [tau_new,adev_new] = gyro_allan_dev(rate_seg{i},dt./3600,500);
        
        tau_seg{i}          = tau_new;
        adev_seg{i}         = adev_new;
 toc
    end
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % PLOT - All Rates Plot
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    
    h_plts(h_ind) = figure;
    center_and_resize_figure(h_plts(h_ind),horz_figs)
    append_num = ['_' sprintf('%02d',h_ind)];
    
    for i = 1:n_splits
        col_ind = mod(i-1,num_pwrs)+1;
        plot_col = cols(col_ind,:);
        plot(t_disp_seg{i},rate_seg{i},'color',plot_col)
        hold on
    end
    
    xlim([min(t_disp) max(t_disp)]);
    
    ylabel('Rate (deg/hr)','fontweight','bold')
    xlabel(xlab_str,'fontweight','bold')
    
    lgd = legend(pwr_leg_str,'Location','EastOutside');
    lgd.Title.String = {'FLS Power';'(dBm)'};
    
    plot_name = 'Normalized Rate at Various Optical Powers';
    plot_file_name = [append_num ' ' plot_name];
    add_name = ['\fontsize{16}' plot_name '\fontsize{10}'];
    tit = [add_name;tit_str];
    title(tit)
    
    grid minor
    a = gca;
    a.MinorGridColor = [0 0 0];
    a.MinorGridAlpha = 0.35000000000000;
    a.MinorGridLineStyle = '-';
    savefig(plot_file_name)
    if sv_pngs
        img = getframe(gcf);
        imwrite(img.cdata,[plot_file_name '.png']);
    end
    
    
    h_ind = h_ind+1;
    
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % PLOT - All Sections Rate Plots
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    for i = 1:n_splits
        seg_pow_ind = mod(i-1,num_pwrs)+1;
        col_ind = mod(i-1,num_pwrs)+1;
        plot_col = cols(col_ind,:);
        
        t_this_seg      = t_disp_seg{i};
        rate_this_seg   = rate_seg{i};
        filt_this_seg   = filter(mv_avg_num,1,rate_this_seg); %averaged data
        filt_this_seg   = filt_this_seg(win_sz:end);
        t_ds_seg        = t_this_seg(win_half:end-win_half);
        t_ds_seg        = t_ds_seg(1:length(filt_this_seg));
        
        h_plts(h_ind) = figure;
        center_and_resize_figure(h_plts(h_ind),horz_figs)
        append_num = ['_' sprintf('%02d',h_ind)];
        h_ind = h_ind+1;
        
        plot(t_this_seg,rate_this_seg,'color','b') %plot_col
        hold on
        plot(t_ds_seg,filt_this_seg,'Color','c')
        xlim([min(t_this_seg) max(t_this_seg)]);
        
        ylabel('Rate (deg/hr)','fontweight','bold')
        xlabel(xlab_str,'fontweight','bold')
        
        leg_str = {[num2str(round(sr)) 'Hz Rate'];[num2str(round(1/ds_rate)) ' sec Avg.']};
        legend(leg_str,'Location',[0.782795140182392   0.797815763080919   0.122395830746326   0.135742183774710]); %,'Orientation','Horizontal')
        
        [rate_stats,add_tit] = get_plot_stats();
        plot_name = ['Normalized Rate - FLS Power = ' pwr_leg_str{seg_pow_ind} 'dBm - Data Section ' num2str(i)];
        file_plot_name = ['Norm Rate - Data Section '  num2str(i)];
        plot_file_name = [append_num ' ' file_plot_name];
        add_name = ['\fontsize{16}' plot_name '\fontsize{10}'];
        tit = [add_name;tit_str;add_tit];
        title(tit)
        grid minor
        a = gca;
        a.MinorGridColor = [0 0 0];
        a.MinorGridAlpha = 0.35000000000000;
        a.MinorGridLineStyle = '-';
        
        savefig(plot_file_name)
        if sv_pngs
            img = getframe(gcf);
            imwrite(img.cdata,[plot_file_name '.png']);
        end
        
        
        
    end
        
        

    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % PLOT - ALL Allan Deviations Together
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  
  
    h_plts(h_ind) = figure;
    center_and_resize_figure
    append_num = ['_' sprintf('%02d',h_ind)];
    h_ind = h_ind+1;
    
    
    for i = 1:n_splits
        col_ind = mod(i-1,num_pwrs)+1;
        plot_col = cols(col_ind,:);
                
        if Allan_sec
            tau_plot = tau_seg{i}.*3600;
        else
            tau_plot = tau_seg{i};
        end
        
        if i == 1
            loglog(tau_plot,adev_seg{i},'Color',plot_col)
            hold on
        else
            plot(tau_plot,adev_seg{i},'Color',plot_col);
        end
    end

    lgd = legend(pwr_leg_str,'Location','Southwest');
    lgd.Title.String = {'FLS Power (dBm)'};
    lgd.NumColumns = 2;
    
    ylabel({'Root Allan Variance';'Rate Uncertainty (deg/hr.)'},'fontweight','bold')
    
    if Allan_sec
        xlabel('Cluster Time (sec.)','fontweight','bold')
    else
        xlabel('Cluster Time (Hrs.)','fontweight','bold')
    end
    
    plot_name = 'All Sections - Root Allan Variance';
    add_name = ['\fontsize{16}' plot_name '\fontsize{10}'];
    plot_file_name = [append_num ' ' plot_name];
    tit = [add_name; tit_str];
    title(tit)
    set(gcf,'Position',[100   100   636   501])
    grid minor
    a = gca;
    a.MinorGridColor = [0 0 0];
    a.MinorGridAlpha = 0.35000000000000;
    a.MinorGridLineStyle = '-';
    savefig(plot_file_name)
    if sv_pngs
        img = getframe(gcf);
        imwrite(img.cdata,[plot_file_name '.png']);
    end
    
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % PLOT - Individual Allan Deviations with ARW Fits
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    for i = 1:n_splits
        
        col_ind = mod(i-1,num_pwrs)+1;
        plot_col = cols(col_ind,:);
        
        tau_this = tau_seg{i};
        
        if Allan_sec
            tau_plot = tau_seg{i}.*3600;
        else
            tau_plot = tau_seg{i};
        end
        
        ad_pl = adev_seg{i};
        
        h_plts(h_ind) = figure;
        center_and_resize_figure
        append_num = ['_' sprintf('%02d',h_ind)];
        h_ind = h_ind+1;
        
        loglog(tau_plot,ad_pl,'Marker','.','Color',plot_col);%,tau,adev_normrate,'c.')
        hold on
        
        inds = find(tau_plot>=ARW_start & tau_plot<=ARW_end);
        t_log = log10(tau_this(inds));
        ad_log = log10(ad_pl(inds));
        B = mean(ad_log)+0.5*mean(t_log);
        ARW = 1e6.*10.^B; %ARW in deg/rt-hr
        hold on
        ARW_fit = 10.^(-0.5.*log10(tau_this)+B);
        
        ARW_line = plot(tau_plot,ARW_fit,'k-','LineWidth',2);
        
        leg_str = {'Rate'; 'ARW Fit'};
        legend(leg_str,'Location','NorthEast');
        
        ylabel({'Root Allan Variance';'Rate Uncertainty (deg/hr.)'},'fontweight','bold')
        
        if Allan_sec
            xlabel('Cluster Time (sec.)','fontweight','bold')
        else
            xlabel('Cluster Time (Hrs.)','fontweight','bold')
        end
        
        add_tit = ['\color{red}ARW = ' num2str(ARW) 'deg/rt-hr.'];      
        plot_name = ['RAV - FLS Power = ' pwr_leg_str{col_ind} 'dBm - Data Section ' num2str(i)];
        add_name = ['\fontsize{16}' plot_name '\fontsize{10}'];
        
        plotnameforfile = ['Root Allan Variance - Data Section '  num2str(i)];
        plot_file_name = [append_num ' ' plotnameforfile];
        tit = [add_name; tit_str;' ';add_tit];
        title(tit)
        set(gcf,'Position',[100   100   636   501])
        grid minor
        a = gca;
        a.MinorGridColor = [0 0 0];
        a.MinorGridAlpha = 0.35000000000000;
        a.MinorGridLineStyle = '-';
        savefig(plot_file_name)
        if sv_pngs
            img = getframe(gcf);
            imwrite(img.cdata,[plot_file_name '.png']);
        end
        
    end

end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% ALL PLOTTING VARIABLES
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% t_disp                    % Elapsed Time for display (sec/min/hrs depending on total run duration
% t_T_disp                  % Elapsed Time from thermal data file

% heater_output             % Heater Output (W) not interpolated to gyro sample spacing
% coil_T_interp             % Coil Pt100 RTD (on or near coil surface) - interpolated to gyro data sample spacing
% igloo_T_interp            % Coil enclosure air Pt100 RTD - interpolated to gyro data sample spacing
% optics_T_interp           % Optics enclosure Pt100 RTD - interpolated to gyro data sample spacing
% lab_T_interp              % Lab air Pt100 RTD - interpolated to gyro data sample spacing
% plate_T_interp            % Heater Controller Plate Thermistor - interpolated to gyro data sample spacing
% air_T_interp              % Heater Controlled Enclosure Air Thermistor - interpolated to gyro data sample spacing

% T_data.T_coil             % Coil Pt100 RTD (on or near coil surface) - interpolated to gyro data sample spacing
% T_data.T_igloo            % Coil enclosure air Pt100 RTD - interpolated to gyro data sample spacing
% T_data.T_optics           % Optics enclosure Pt100 RTD - interpolated to gyro data sample spacing
% T_data.T_lab              % Lab air Pt100 RTD - interpolated to gyro data sample spacing
% T_data.T_plate            % Heater Controller Plate Thermistor - interpolated to gyro data sample spacing
% T_data.T_air              % Heater Controlled Enclosure Air Thermistor - interpolated to gyro data sample spacing
 
% raw_rate                  % 1st harmonic demod in-phase component X(Vrms)
% rate_quaderr              % 1st harmonic demod quadrature component Y(Vrms)

% int_x                     % 2nd harmonic demod in-phase component X(Vrms)
% int_y                     % 2nd harmonic demod quadrature component Y(Vrms)
% int                       % 2nd harmonic demod magnitude R (Vrms)


% R                         % Raw demod word (1st harm / 2nd harm)
% R_ds                      % Filtered demod word
% t_ds                      % time samples for filtered rate
% rate                      % Calculated Rate Signal (deg/hr)
% rate_no_norm              % Rate Signal without normalization to 2nd harmonic signal
% rate_filt                 % Moving average filtered Rate 
% T_filt                    % Moving average filtered primary temperature signal

% T_fit_rate                % Fitted rate signal (offset, temperature, T_dot)
% T_fit_res                 % Rate signal with T-dot fit subtracted (offset, temperature, T_dot)
% T_coeffs                  % Fit coefficients from T-Dot fitting

% tau                       % Time clusters (hrs.) for rate RAV
% adev_normrate             % Rate RAV
% adev_rate                 % Rate RAV without normalization to 2nd harmonic
% tau_filt                  % Time clusters (hrs.) for moving average filtered rate RAV
% adev_rate_filt            % Moving average filtered Rate RAV
% adev_fit_res              % RAV of the moving average filtered rate signal with the T-Dot fitted componented subtracted

% freq                      % Frequency from Rate PSD calculation
% psd_rate                  % Rate PSD
% freq_T                    % Frequency from temperature PSD calculation - based on linear interpolated temperature data

% psd_coil                  % Coil Surface Pt100 RTD power spectral density
% psd_igloo                 % Igloo or Coil environs Pt100 RTD power spectral density
% psd_optics                % Optics enclosure Pt100 RTD power spectral density
% psd_air                   % Lab Air Pt100 RTD power spectral density
% psd_plate                 % Blue enclosure plate thermistor (control) power spectral density
% psd_enc                   % Blue enclosure air thermistor power spectral density

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PLOT - 1st Harmonic In-Phase Raw Data and 2nd Harmonic Magnitude Raw Data
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
h_plts(h_ind) = figure;
center_and_resize_figure(h_plts(h_ind),horz_figs)
append_num = ['_' sprintf('%02d',h_ind)];
 
yyaxis left
plot(t_disp,raw_rate.*1e6);
ylabel('Raw Rate Demod - 1st Harmonic X (V)','fontweight','bold')

yyaxis right
plot(t_disp,int.*1e3);
ylabel('Raw Intensity Demod - 2nd Harmonic R (mV)','fontweight','bold')

xlim([min(t_disp) max(t_disp)]);
xlabel(xlab_str,'fontweight','bold')

leg_str = {[num2str(round(sr)) 'Hz Rate'];['Intensity']};
legend(leg_str,'Location','best'); %,'Orientation','Horizontal')

[raw_stats,add_tit] = get_plot_stats();

plot_name = 'Raw Rate and Intensity Data';
plot_file_name = [append_num ' ' plot_name];

add_name = ['\fontsize{16}' plot_name '\fontsize{10}'];
tit = [add_name;tit_str;add_tit];
title(tit)
grid minor
a = gca;
a.MinorGridColor = [0 0 0];
a.MinorGridAlpha = 0.35000000000000;
a.MinorGridLineStyle = '-';
savefig(plot_file_name)
if sv_pngs
    img = getframe(gcf);
    imwrite(img.cdata,[plot_file_name '.png']);
end
h_ind = h_ind+1;



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PLOT - Demod Word
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
h_plts(h_ind) = figure;
center_and_resize_figure(h_plts(h_ind),horz_figs)
append_num = ['_' sprintf('%02d',h_ind)];

plot(t_disp,R,'color','b')
%[ 0.850980392156863   0.325490196078431   0.098039215686275]
%[1.000000000000000   0.600000000000000   0.784313725490196]
hold on
plot(t_ds,R_ds,'Color','c')
xlim([min(t_disp) max(t_disp)]);

ylabel('Rate Demod Word - (Rate X / Int)','fontweight','bold')
xlabel(xlab_str,'fontweight','bold')

leg_str = {[num2str(round(sr)) 'Hz Rate'];[num2str(round(1/ds_rate)) ' sec Avg.']};
legend(leg_str,'Location','best'); %,'Orientation','Horizontal')

[demod_stats,add_tit] = get_plot_stats();

plot_name = 'Intensity Normalised Fractional Demod Word';
plot_file_name = [append_num ' ' plot_name];
add_name = ['\fontsize{16}' plot_name '\fontsize{10}'];
tit = [add_name;tit_str;add_tit];
title(tit)
grid minor
a = gca;
a.MinorGridColor = [0 0 0];
a.MinorGridAlpha = 0.35000000000000;
a.MinorGridLineStyle = '-';
savefig(plot_file_name) 
if sv_pngs
    img = getframe(gcf);
    imwrite(img.cdata,[plot_file_name '.png']);
end
h_ind = h_ind+1;
a = gca;

demod_stats(1).name
demod_stats(1).mean
demod_stats(2).name
demod_stats(2).mean

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PLOT - Fractional Rate and filtered
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

h_plts(h_ind) = figure;
center_and_resize_figure(h_plts(h_ind),horz_figs)
append_num = ['_' sprintf('%02d',h_ind)];

if use_thermal_data    
    yyaxis left
    plot(t_disp,rate,'b-')
    %[ 0.850980392156863   0.325490196078431   0.098039215686275]
    %[1.000000000000000   0.600000000000000   0.784313725490196]
    hold on
    plot(t_ds,rate_filt,'c-')
    ylabel('Rate (deg/hr)','fontweight','bold')
    xlabel(xlab_str,'fontweight','bold')
    
    xlim([min(t_disp) max(t_disp)]);
       
    leg_str = {[num2str(round(sr)) 'Hz Rate'];[num2str(round(1/ds_rate)) ' sec Avg.']; 'Plate Temperature'; 'Air Temperature'};
    legend(leg_str,'Location','best'); %,'Orientation','Horizontal')
    [rate_stats,add_tit] = get_plot_stats();
    
    yyaxis right
    plot(t_disp,main_T     ,'color',[0.850980392156863   0.325490196078431   0.098039215686275],'linestyle','-','linewidth',1)
    hold on
    if show_all_Ts
        plot(t_disp,coil_T_interp     ,'color',[ 0.850980392156863   0.325490196078431   0.098039215686275],'linestyle','-','linewidth',1)
        plot(t_disp,igloo_T_interp    ,'color',[1 0.6 0.784]          ,'linestyle','-','linewidth',1,'Marker','None')
        plot(t_disp,optics_T_interp   ,'color',[0.67   0.11   0.43]   ,'linestyle','-','linewidth',1,'Marker','None')
        plot(t_disp,lab_T_interp      ,'color',[0.81   0.59   0.15]   ,'linestyle','-','linewidth',1,'Marker','None')
        plot(t_disp,plate_T_interp         ,'color',[0   0   0.3]         ,'linestyle','-','linewidth',1,'Marker','None')
        plot(t_disp,air_T_interp           ,'color',[0   0.75   0]         ,'linestyle','-','linewidth',1,'Marker','None')
    end
    
    ylabel('Temperature (C)','fontweight','bold')
    
    leg_str = {[num2str(round(sr)) 'Hz Rate'];[num2str(round(1/ds_rate)) ' sec Avg.'];...
        'Coil T'; 'Enclosure T'; 'Optics T';'Lab Air T'; 'Plate T'; 'Enclosure air Therm'};
    legend(leg_str,'Location',[0.782795140182392   0.797815763080919   0.122395830746326   0.135742183774710]); %,'Orientation','Horizontal')
    
    plot_name = 'Intensity Normalized Rotation Rate';
    plot_file_name = [append_num ' ' plot_name];
    add_name = ['\fontsize{16}' plot_name '\fontsize{10}'];
    tit = [add_name;tit_str;add_tit];
    title(tit)
    grid minor
    a = gca;
    a.MinorGridColor = [0 0 0];
    a.MinorGridAlpha = 0.35000000000000;
    a.MinorGridLineStyle = '-';
    yyaxis left
    savefig(plot_file_name)
    if sv_pngs
        img = getframe(gcf);
        imwrite(img.cdata,[plot_file_name '.png']);
    end
   
else

    plot(t_disp,rate,'color','b')
    hold on
    plot(t_ds,rate_filt,'Color','c')
    xlim([min(t_disp) max(t_disp)]);
    
    ylabel('Rate (deg/hr)','fontweight','bold')
    xlabel(xlab_str,'fontweight','bold')
    
    leg_str = {[num2str(round(sr)) 'Hz Rate'];[num2str(round(1/ds_rate)) ' sec Avg.']};
    legend(leg_str,'Location',[0.782795140182392   0.797815763080919   0.122395830746326   0.135742183774710]); %,'Orientation','Horizontal')
    
    [rate_stats,add_tit] = get_plot_stats();
    plot_name = 'Intensity Normalized Rotation Rate';
    plot_file_name = [append_num ' ' plot_name];
    add_name = ['\fontsize{16}' plot_name '\fontsize{10}'];
    tit = [add_name;tit_str;add_tit];
    title(tit)
    grid minor
    a = gca;
    a.MinorGridColor = [0 0 0];
    a.MinorGridAlpha = 0.35000000000000;
    a.MinorGridLineStyle = '-';
    savefig(plot_file_name)
    if sv_pngs
        img = getframe(gcf);
        imwrite(img.cdata,[plot_file_name '.png']);
    end
end

h_ind = h_ind+1;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PLOT - Rate vs. Heater Output
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if use_thermal_data

    h_plts(h_ind) = figure;
    center_and_resize_figure(h_plts(h_ind),horz_figs)
    append_num = ['_' sprintf('%02d',h_ind)];
   
    yyaxis left
    plot(t_disp,rate,'b-')
    %[ 0.850980392156863   0.325490196078431   0.098039215686275]
    %[1.000000000000000   0.600000000000000   0.784313725490196]
    hold on
    plot(t_ds,rate_filt,'c-')
    ylabel('Rate (deg/hr)','fontweight','bold')
    xlabel(xlab_str,'fontweight','bold')
    
    xlim([min(t_disp) max(t_disp)]);
       
    leg_str = {[num2str(round(sr)) 'Hz Rate'];[num2str(round(1/ds_rate)) ' sec Avg.']; 'Heater Output'};
    legend(leg_str,'Location','best'); %,'Orientation','Horizontal')
    [rate_stats,add_tit] = get_plot_stats();
    
    yyaxis right
    plot(t_T_disp,heater_output     ,'color',[ 0.850980392156863   0.325490196078431   0.098039215686275],'linestyle','-','linewidth',1)
    hold on
       
    leg_str = {[num2str(round(sr)) 'Hz Rate'];[num2str(round(1/ds_rate)) ' sec Avg.'];...
        'Heater Output'};
    legend(leg_str,'Location',[0.782795140182392   0.797815763080919   0.122395830746326   0.135742183774710]); %,'Orientation','Horizontal')
    
    plot_name = 'Norm Rate vs Heater';
    plot_file_name = [append_num ' ' plot_name];
    add_name = ['\fontsize{16}' plot_name '\fontsize{10}'];
    tit = [add_name;tit_str;add_tit];
    title(tit)
    grid minor
    a = gca;
    a.MinorGridColor = [0 0 0];
    a.MinorGridAlpha = 0.35000000000000;
    a.MinorGridLineStyle = '-';
    yyaxis left
    savefig(plot_file_name)
    if sv_pngs
        img = getframe(gcf);
        imwrite(img.cdata,[plot_file_name '.png']);
    end
end
h_ind = h_ind+1;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PLOT - Fitted Rate Plot
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if use_thermal_data & fit_T_dot
    h_plts(h_ind) = figure;
    center_and_resize_figure(h_plts(h_ind),horz_figs)
    append_num = ['_' sprintf('%02d',h_ind)];
    yyaxis left
    plot(t_ds,rate_filt,'b-')
    hold on
    plot(t_ds,T_fit_rate,'c-')
        
    ylabel({'Rate (deg/hr)'},'fontweight','bold')
    xlabel(xlab_str,'fontweight','bold')
    
    
    xlim([min(t_disp) max(t_disp)]);

    yyaxis right
    plot(t_ds,T_filt,'color',[ 0.850980392156863   0.325490196078431   0.098039215686275],'linestyle','-','linewidth',2)
    ylabel('Coil Temperature (C)','fontweight','bold')
            
    plot_name = 'T-Dot Fitted Rate';
    plot_file_name = [append_num ' ' plot_name];
    add_name = ['\fontsize{16}' plot_name '\fontsize{10}'];
    fit_coeffs_str = ['\fontsize{12}Fit Coefficients: Offset = ' num2str(T_coeffs(1)) ' deg/hr  |  T = ' num2str(T_coeffs(2)) ' deg/hr/C  |  \color{blue}T-Dot = ' num2str(T_coeffs(3)) ' deg/hr/(C/min)'];
        
    tit = [add_name;tit_str;' ';fit_coeffs_str];
    title(tit)
    
    leg_str = {[num2str(round(1/ds_rate)) ' sec Avg. Rate']; 'Fitted Rate'; [num2str(round(1/ds_rate)) 's avg. Coil Temperature']};
    legend(leg_str,'Location',[0.786267362870276 0.834925138080919 0.153645829815004 0.135742183774710]); %,'Orientation','Horizontal')
    
    grid minor
    a = gca;
    a.MinorGridColor = [0 0 0];
    a.MinorGridAlpha = 0.35000000000000;
    a.MinorGridLineStyle = '-';
    savefig(plot_file_name)
    if sv_pngs
        img = getframe(gcf);
        imwrite(img.cdata,[plot_file_name '.png']);
    end
end
h_ind = h_ind+1;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PLOT - Fitted Rate Plot - 2nd DERIVATIVE T-DOT
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if use_thermal_data & fit_T_dot
    h_plts(h_ind) = figure;
    center_and_resize_figure(h_plts(h_ind),horz_figs)
    append_num = ['_' sprintf('%02d',h_ind)];
    yyaxis left
    plot(t_ds,rate_filt,'b-')
    hold on
    plot(t_ds(1:end-2),Tdd_fit_rate,'c-')
        
    ylabel({'Rate (deg/hr)'},'fontweight','bold')
    xlabel(xlab_str,'fontweight','bold')
    
    
    xlim([min(t_disp) max(t_disp)]);

    yyaxis right
    plot(t_ds,T_filt,'color',[ 0.850980392156863   0.325490196078431   0.098039215686275],'linestyle','-','linewidth',2)
    ylabel('Coil Temperature (C)','fontweight','bold')
            
    plot_name = '2nd Order T-Dot Fitted Rate';
    plot_file_name = [append_num ' ' plot_name];
    add_name = ['\fontsize{16}' plot_name '\fontsize{10}'];
    fit_coeffs_str = ['\fontsize{12}Fit Coefficients: Offset = ' num2str(Tdd_coeffs(1),'%0.2f') ' deg/hr  |  T = ' num2str(Tdd_coeffs(2),'%0.2g') ' deg/hr/C  |  \color{blue}T-Dot = ' num2str(Tdd_coeffs(3),'%0.3f') ' deg/hr/(C/min) | \color{black}T-Dotdot = ' num2str(Tdd_coeffs(4),'%0.3f') ' deg/hr/(C/min)^2'];
        
    tit = [add_name;tit_str;' ';fit_coeffs_str];
    title(tit)
    
    leg_str = {[num2str(round(1/ds_rate)) ' sec Avg. Rate']; 'Fitted Rate'; [num2str(round(1/ds_rate)) 's avg. Coil Temperature']};
    legend(leg_str,'Location',[0.786267362870276 0.834925138080919 0.153645829815004 0.135742183774710]); %,'Orientation','Horizontal')
    
    grid minor
    a = gca;
    a.MinorGridColor = [0 0 0];
    a.MinorGridAlpha = 0.35000000000000;
    a.MinorGridLineStyle = '-';
    savefig(plot_file_name)
    if sv_pngs
        img = getframe(gcf);
        imwrite(img.cdata,[plot_file_name '.png']);
    end
end
h_ind = h_ind+1;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PLOT - Rate Residual
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if use_thermal_data & fit_T_dot
    
    h_plts(h_ind) = figure;
    center_and_resize_figure(h_plts(h_ind),horz_figs)
    append_num = ['_' sprintf('%02d',h_ind)];
    yyaxis left
    plot(t_ds,T_fit_res)

    ylabel({'Fit Residual (deg/hr)'},'fontweight','bold')
    xlabel(xlab_str,'fontweight','bold')
    
    
    xlim([min(t_disp) max(t_disp)]);

    yyaxis right
    plot(t_ds,T_filt)
    ylabel('Air Temperature (C)','fontweight','bold')
            
    plot_name = 'T-Dot Fit Residual';
    plot_file_name = [append_num ' ' plot_name];
    add_name = ['\fontsize{16}' plot_name '\fontsize{10}'];
            
    tit = [add_name;tit_str];
    title(tit)

    grid minor
    a = gca;
    a.MinorGridColor = [0 0 0];
    a.MinorGridAlpha = 0.35000000000000;
    a.MinorGridLineStyle = '-';
    savefig(plot_file_name)
    if sv_pngs
        img = getframe(gcf);
        imwrite(img.cdata,[plot_file_name '.png']);
    end
end
h_ind = h_ind+1;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PLOT - Fit vs. Temperature
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if use_thermal_data & fit_T_dot
    h_plts(h_ind) = figure;
    center_and_resize_figure(h_plts(h_ind),[0.65 0.65])
    append_num = ['_' sprintf('%02d',h_ind)];
        
    
    plot(T_filt,rate_filt,'b')
    hold on
    plot(T_filt,T_fit_rate,'c-')
    
    ylabel({'Rate (deg/hr)'},'fontweight','bold')
    xlabel('\color{red}Coil Temperature','fontweight','bold')
            
    plot_name = 'Fit vs Temperature';
    plot_file_name = [append_num ' ' plot_name];
    add_name = ['\fontsize{16}' plot_name '\fontsize{10}'];
            
    tit = [add_name;tit_str];
    title(tit)
    
    leg_str = {[num2str(round(1/ds_rate)) ' sec Avg. Rate']; 'Fitted Rate'};
    legend(leg_str,'Location','Best')
    
    grid minor
    a = gca;
    a.MinorGridColor = [0 0 0];
    a.MinorGridAlpha = 0.35000000000000;
    a.MinorGridLineStyle = '-';
    savefig(plot_file_name)
    if sv_pngs
        img = getframe(gcf);
        imwrite(img.cdata,[plot_file_name '.png']);
    end
end
h_ind = h_ind+1;
    
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PLOT - Rate vs. Temperature
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if use_thermal_data & fit_T_dot
    
    h_plts(h_ind) = figure;
    center_and_resize_figure(h_plts(h_ind),[0.65 0.65])
    append_num = ['_' sprintf('%02d',h_ind)];
    h_ind = h_ind+1;
    
    if show_int_vs_T
        yyaxis left
        plot(main_T,rate,'b-')
        ylabel({'Rate (deg/hr)'},'fontweight','bold')
        yyaxis right
        plot(main_T,int,'r-')
        ylabel({'2nd Harmonic Intensity (Vrms)'},'fontweight','bold')
    else
        plot(main_T,rate,'b-')
        ylabel({'Rate (deg/hr)'},'fontweight','bold')
    end
   
    xlabel('\color{red}Coil Temperature','fontweight','bold')
            
    legend(leg_str,'Location','Best')
    
    plot_name = 'Rate vs Temperature';
    plot_file_name = [append_num ' ' plot_name];
    add_name = ['\fontsize{16}' plot_name '\fontsize{10}'];        
    tit = [add_name;tit_str];
    title(tit)

    grid minor
    a = gca;
    a.MinorGridColor = [0 0 0];
    a.MinorGridAlpha = 0.35000000000000;
    a.MinorGridLineStyle = '-';
    savefig(plot_file_name)
    if sv_pngs
        img = getframe(gcf);
        imwrite(img.cdata,[plot_file_name '.png']);
    end
end
h_ind = h_ind+1;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PLOT - Filtered Rate vs. Temperature
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if use_thermal_data & fit_T_dot
    
    h_plts(h_ind) = figure;
    center_and_resize_figure(h_plts(h_ind),[0.65 0.65])
    append_num = ['_' sprintf('%02d',h_ind)];
    h_ind = h_ind+1;
    
        
    if show_int_vs_T
        yyaxis left
        plot(T_filt,rate_filt,'b-')
        ylabel({'Filtered Rate (deg/hr)'},'fontweight','bold')
        yyaxis right
        plot(T_filt,int_filt,'r-')
        ylabel({'Filtered 2nd Harmonic Intensity (Vrms)'},'fontweight','bold')
    else
        plot(T_filt,rate_filt,'b-')
        ylabel({'Rate (deg/hr)'},'fontweight','bold')
    end
   
    xlabel('\color{red}Filtered Coil Temperature (C)','fontweight','bold')
            
    legend(leg_str,'Location','Best')
    
    plot_name = 'Filtered Rate vs Temperature';
    plot_file_name = [append_num ' ' plot_name];
    add_name = ['\fontsize{16}' plot_name '\fontsize{10}'];        
    tit = [add_name;tit_str];
    title(tit)

    grid minor
    a = gca;
    a.MinorGridColor = [0 0 0];
    a.MinorGridAlpha = 0.35000000000000;
    a.MinorGridLineStyle = '-';
    savefig(plot_file_name)
    if sv_pngs
        img = getframe(gcf);
        imwrite(img.cdata,[plot_file_name '.png']);
    end
end
h_ind = h_ind+1;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PLOT - Root Allan Variance
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if Allan_sec
    tau_plot = tau.*3600;
else
    tau_plot = tau;
end
    
h_plts(h_ind) = figure;
center_and_resize_figure
append_num = ['_' sprintf('%02d',h_ind)];
h_ind = h_ind+1;

loglog(tau_plot,adev_rate,'k.');%,tau,adev_normrate,'c.')
hold on
plot(tau_plot,adev_normrate,'b.');

if fit_T_dot
    plot(tau_filt,adev_rate_filt,'c.');
    plot(tau_filt,adev_fit_res,'g.');
end

inds = find(tau_plot>=ARW_start & tau_plot<=ARW_end);
t_log = log10(tau(inds));
ad_log = log10(adev_normrate(inds));
B = mean(ad_log)+0.5*mean(t_log);
ARW = 1e6.*10.^B; %ARW in deg/rt-hr
hold on
ARW_fit = 10.^(-0.5.*log10(tau)+B);
ARW_line = plot(tau_plot,ARW_fit,'r');

if use_thermal_data
    leg_str = {'Un-normalized Rate';'Intensity Normalized Rate'; 'Moving Avg. Rate';'T-Dot Fit Residual'; 'ARW Fit'};
else
    leg_str = {'Un-normalized Rate';'Intensity Normalized Rate'; 'ARW Fit'};
end

legend(leg_str,'Location','NorthEast'); %,'Orientation','Horizontal')

ylabel({'Root Allan Variance';'Rate Uncertainty (deg/hr.)'},'fontweight','bold')
if Allan_sec
    xlabel('Cluster Time (sec.)','fontweight','bold')
else
    xlabel('Cluster Time (Hrs.)','fontweight','bold')
end
    

add_tit = ['\color{red}ARW = ' num2str(ARW) 'deg/rt-hr.'];
plot_name = 'Root Allan Variance';
add_name = ['\fontsize{16}' plot_name '\fontsize{10}'];
plot_file_name = [append_num ' ' plot_name];
tit = [add_name; tit_str;' ';add_tit];
title(tit)
set(gcf,'Position',[100   100   636   501])
grid minor
a = gca;
a.MinorGridColor = [0 0 0];
a.MinorGridAlpha = 0.35000000000000;
a.MinorGridLineStyle = '-';
savefig(plot_file_name)
if sv_pngs
    img = getframe(gcf);
    imwrite(img.cdata,[plot_file_name '.png']);
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PLOT - Rate PSD
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

h_plts(h_ind) = figure;
center_and_resize_figure
append_num = ['_' sprintf('%02d',h_ind)];
h_ind = h_ind+1;

loglog(freq,psd_rate,'b');%,tau,adev_normrate,'c.')

%leg_str = {'Intensity Normalized Rate'; 'ARW Fit'};
%legend(leg_str,'Location','NorthEast'); %,'Orientation','Horizontal')

ylabel('Rate Power Spectrum (deg/hr)^2/Hz','fontweight','bold')
xlabel('Frequency (Hz)','fontweight','bold')

%add_tit = ['\color{red}ARW = ' num2str(ARW) 'deg/rt-hr.'];
plot_name = 'Rate Power Spectral Density';
plot_file_name = [append_num ' ' plot_name];
add_name = ['\fontsize{16}' plot_name '\fontsize{10}'];
tit = [add_name; tit_str;' ';add_tit];
title(tit)
set(gcf,'Position',[100   100   636   501])
grid minor
a = gca;
a.MinorGridColor = [0 0 0];
a.MinorGridAlpha = 0.35000000000000;
a.MinorGridLineStyle = '-';
savefig(plot_file_name)
if sv_pngs
    img = getframe(gcf);
    imwrite(img.cdata,[plot_file_name '.png']);
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PLOT - 1st and 2nd harmonic raw PSD
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

h_plts(h_ind) = figure;
center_and_resize_figure
append_num = ['_' sprintf('%02d',h_ind)];
h_ind = h_ind+1;


loglog(freq,psd_1stharm);%,tau,adev_normrate,'c.')
hold on
loglog(freq,psd_2ndharm);

ylabel('Demod Voltage Spectrum V^2/Hz','fontweight','bold')
xlabel('Frequency (Hz)','fontweight','bold')
legend({'1st Harmonic';'2nd Harmonic'},'location','best')

%add_tit = ['\color{red}ARW = ' num2str(ARW) 'deg/rt-hr.'];
plot_name = 'Demod PSDs';
plot_file_name = [append_num ' ' plot_name];
add_name = ['\fontsize{16}' plot_name '\fontsize{10}'];
tit = [add_name; tit_str;' ';add_tit];
title(tit)
set(gcf,'Position',[100   100   636   501])
grid minor
a = gca;
a.MinorGridColor = [0 0 0];
a.MinorGridAlpha = 0.35000000000000;
a.MinorGridLineStyle = '-';
savefig(plot_file_name)
if sv_pngs
    img = getframe(gcf);
    imwrite(img.cdata,[plot_file_name '.png']);
end
h_ind = h_ind+1;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PLOT - Temperature PSD
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if use_thermal_data
    h_plts(h_ind) = figure;
    center_and_resize_figure
    append_num = ['_' sprintf('%02d',h_ind)];

    
    loglog(freq_T,psd_coil     ,'color',[ 0.850980392156863   0.325490196078431   0.098039215686275],'linestyle','-','linewidth',1)
    hold on
    
    loglog(freq_T,psd_igloo    ,'color',[1 0.6 0.784]          ,'linestyle','-','linewidth',1,'Marker','None')
    loglog(freq_T,psd_optics   ,'color',[0.67   0.11   0.43]   ,'linestyle','-','linewidth',1,'Marker','None')
    loglog(freq_T,psd_air      ,'color',[0.81   0.59   0.15]   ,'linestyle','-','linewidth',1,'Marker','None')
    loglog(freq_T,psd_plate    ,'color',[0   0   0.3]          ,'linestyle','-','linewidth',1,'Marker','None')
    loglog(freq_T,psd_enc      ,'color',[0   0.75   0]         ,'linestyle','-','linewidth',1,'Marker','None')
    
    
    leg_str = {'Coil RTD'; 'Encl. RTD'; 'Optics RTD';'Lab RTD'; 'Plate Therm'; 'Encl. Therm'};
    legend(leg_str,'Location','NorthEast'); %,'Orientation','Horizontal')
    
    ylabel('Temperature Power Spectrum (C^2/Hz)','fontweight','bold')
    xlabel('Frequency (Hz)','fontweight','bold')
    
    %add_tit = ['\color{red}ARW = ' num2str(ARW) 'deg/rt-hr.'];
    plot_name = 'Temperature PSD';
    plot_file_name = [append_num ' ' plot_name];
    add_name = ['\fontsize{16}' plot_name '\fontsize{10}'];
    tit = [add_name; tit_str;' ';add_tit];
    title(tit)
    set(gcf,'Position',[100   100   636   501])
    grid minor
    a = gca;
    a.MinorGridColor = [0 0 0];
    a.MinorGridAlpha = 0.35000000000000;
    a.MinorGridLineStyle = '-';
    savefig(plot_file_name)
    if sv_pngs
        img = getframe(gcf);
        imwrite(img.cdata,[plot_file_name '.png']);
    end
end
h_ind = h_ind+1;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PLOT - Filtered and Interpolated Temperatures & Saving temperature csv
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if use_thermal_data & show_all_Ts
    h_plts(h_ind) = figure;
    center_and_resize_figure(h_plts(h_ind),horz_figs)
    append_num = ['_' sprintf('%02d',h_ind)];
    
    t_span_TempFilter = 80; %Seconds over which Savitsky-Golay filtering is applied to temperature data
    sgolay_poly = 3;        %Order of Savitsky Golay Filter
    sgolay_sz = floor(floor(t_span_TempFilter*sr)/2).*2+1;
    coil_T_filt = sgolayfilt(main_T,sgolay_poly,sgolay_sz); %averaged data
        
    plot(t_T./60,T_data.T_coil,'Linestyle',':','Color','r')
    hold on
    plot(t_sec./60,coil_T_interp,'Linestyle','none','Marker','o','MarkerEdgeColor','r')
    plot(t_sec./60,coil_T_filt,'Linestyle','-','Linewidth',1.5,'Color','r')
    
    data_for_file = [t_sec coil_T_filt+273.15]';
    fid = fopen('RTD Data for COMSOL.txt','w');
    fprintf(fid,'%10s \t %10s\n','Time (sec)','T (degK)');
    fprintf(fid,'%0.4f \t %0.7e\n',data_for_file);
    fclose(fid);
    
    plot(t_T./60,T_data.T_igloo,'Linestyle','-','Color','b')
    plot(t_sec./60,igloo_T_interp,'Linestyle','none','Marker','o','MarkerEdgeColor','b')
    
    plot(t_T./60,T_data.T_optics,'Linestyle','-','Color','m')
    plot(t_sec./60,optics_T_interp,'Linestyle','none','Marker','o','MarkerEdgeColor','m')
    
    plot(t_T./60,T_data.T_lab,'Linestyle','-','Color',[0 .5 0])
    plot(t_sec./60,lab_T_interp,'Linestyle','none','Marker','o','MarkerEdgeColor',[0 0.5 0])
    
    plot(t_T./60,T_data.T_plate,'Linestyle','-','Color','c')
    plot(t_sec./60,plate_T_interp,'Linestyle','none','Marker','o','MarkerEdgeColor','c')
    
    plot(t_T./60,T_data.T_air,'Linestyle','-','Color','k')
    plot(t_sec./60,air_T_interp,'Linestyle','none','Marker','o','MarkerEdgeColor','k')
    
    xlim([min(t_T./60) max(t_T./60)]);
    xlabel('Elapsed Time (sec)','fontweight','bold')
    
    ylabel('Temperature (C)','fontweight','bold')
    leg_str = {'Coil Pt100'; 'Enclosure Pt100'; 'Optics Pt100';'Lab Air Pt100'; 'Plate Therm'; 'Enclos. Therm'};
    legend(leg_str,'Location','best'); %,'Orientation','Horizontal')
      
    plot_name = 'Temperatures';
    plot_file_name = [append_num ' ' plot_name];
    
    add_name = ['\fontsize{16}' plot_name '\fontsize{10}'];
    tit = [add_name;tit_str;add_tit];
    title(tit)
    grid minor
    a = gca;
    a.MinorGridColor = [0 0 0];
    a.MinorGridAlpha = 0.35000000000000;
    a.MinorGridLineStyle = '-';
    savefig(plot_file_name)
    if sv_pngs
        img = getframe(gcf);
        imwrite(img.cdata,[plot_file_name '.png']);
    end
end
h_ind = h_ind+1;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% PLOT - Temperature Allan Variance
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if use_thermal_data
    
h_plts(h_ind) = figure;
center_and_resize_figure
append_num = ['_' sprintf('%02d',h_ind)];

loglog(tau_T,adev_coil,'r')
hold on
plot(tau_T,adev_igloo   ,'color'    ,[1 0.6 0.784]);
plot(tau_T,adev_optics  ,'color'    ,[0.67   0.11   0.43]);
plot(tau_T,adev_air     ,'color'    ,[0.81   0.59   0.15]);
plot(tau_T,adev_plate   ,'color'    ,[0   0   0.3] );
plot(tau_T,adev_enc     ,'color'    ,[0   0.75   0]);

leg_str = {'Coil RTD'; 'Encl. RTD'; 'Optics RTD';'Lab RTD'; 'Plate Therm'; 'Encl. Therm'};

legend(leg_str,'Location','NorthEast'); %,'Orientation','Horizontal')

ylabel({'Root Allan Variance';'Rate Uncertainty (deg/hr.)'},'fontweight','bold')
xlabel('Cluster Time (Hrs.)','fontweight','bold')

plot_name = 'Temperature RAV';
add_name = ['\fontsize{16}' plot_name '\fontsize{10}'];
plot_file_name = [append_num ' ' plot_name];
tit = [add_name; tit_str;' ';add_tit];
title(tit)
set(gcf,'Position',[100   100   636   501])
grid minor
a = gca;
a.MinorGridColor = [0 0 0];
a.MinorGridAlpha = 0.35000000000000;
a.MinorGridLineStyle = '-';
savefig(plot_file_name)
if sv_pngs
    img = getframe(gcf);
    imwrite(img.cdata,[plot_file_name '.png']);
end
end
h_ind = h_ind+1;


end

